<?php

namespace Tests\Feature\Api;

use App\Models\Category;
use App\Models\Notification;
use App\Models\NotificationOccurrence;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class OccurrenceTest extends TestCase
{
    use RefreshDatabase;

    private User $user;
    private Notification $notification;
    private NotificationOccurrence $occurrence;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create(['email_verified_at' => now()]);
        $this->seed(\Database\Seeders\CategorySeeder::class);

        $category = Category::where('is_default', true)->first();
        $this->notification = Notification::factory()->create([
            'user_id' => $this->user->id,
            'category_id' => $category->id,
        ]);
        $this->occurrence = NotificationOccurrence::create([
            'notification_id' => $this->notification->id,
            'scheduled_at' => now()->addHour(),
            'status' => 'pending',
        ]);
    }

    public function test_can_list_occurrences(): void
    {
        $response = $this->actingAs($this->user)->getJson('/api/occurrences');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'occurrences',
                'meta' => ['current_page', 'last_page', 'per_page', 'total'],
            ]);
    }

    public function test_can_filter_occurrences_by_status(): void
    {
        $response = $this->actingAs($this->user)
            ->getJson('/api/occurrences?status=pending');

        $response->assertStatus(200);
        $this->assertGreaterThanOrEqual(1, $response->json('meta.total'));
    }

    public function test_can_filter_occurrences_by_date_range(): void
    {
        $response = $this->actingAs($this->user)->getJson(
            '/api/occurrences?from_date=' . now()->format('Y-m-d') . '&to_date=' . now()->addDays(1)->format('Y-m-d')
        );

        $response->assertStatus(200);
    }

    public function test_can_complete_occurrence(): void
    {
        $response = $this->actingAs($this->user)
            ->patchJson("/api/occurrences/{$this->occurrence->id}/complete");

        $response->assertStatus(200);
        $this->assertEquals('completed', $this->occurrence->fresh()->status);
        $this->assertNotNull($this->occurrence->fresh()->acted_at);
    }

    public function test_can_dismiss_occurrence(): void
    {
        $response = $this->actingAs($this->user)
            ->patchJson("/api/occurrences/{$this->occurrence->id}/dismiss");

        $response->assertStatus(200);
        $this->assertEquals('dismissed', $this->occurrence->fresh()->status);
    }

    public function test_can_snooze_occurrence(): void
    {
        $response = $this->actingAs($this->user)
            ->patchJson("/api/occurrences/{$this->occurrence->id}/snooze", [
                'duration_minutes' => 15,
            ]);

        $response->assertStatus(200);
        $this->assertEquals('snoozed', $this->occurrence->fresh()->status);
        $this->assertNotNull($this->occurrence->fresh()->snoozed_until);
    }

    public function test_snooze_validates_duration(): void
    {
        $response = $this->actingAs($this->user)
            ->patchJson("/api/occurrences/{$this->occurrence->id}/snooze", [
                'duration_minutes' => 999,
            ]);

        $response->assertStatus(422);
    }

    public function test_can_skip_occurrence(): void
    {
        $response = $this->actingAs($this->user)
            ->patchJson("/api/occurrences/{$this->occurrence->id}/skip");

        $response->assertStatus(200);
        $this->assertEquals('skipped', $this->occurrence->fresh()->status);
    }

    public function test_cannot_modify_other_users_occurrence(): void
    {
        $otherUser = User::factory()->create();

        $response = $this->actingAs($otherUser)
            ->patchJson("/api/occurrences/{$this->occurrence->id}/complete");

        $response->assertStatus(403);
    }
}
