<?php

namespace Tests\Feature\Api;

use App\Models\Category;
use App\Models\Notification;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class NotificationTest extends TestCase
{
    use RefreshDatabase;

    private User $user;
    private Category $category;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create(['email_verified_at' => now()]);
        $this->seed(\Database\Seeders\CategorySeeder::class);
        $this->category = Category::where('is_default', true)->first();
    }

    public function test_can_create_one_time_notification(): void
    {
        $response = $this->actingAs($this->user)->postJson('/api/notifications', [
            'category_id' => $this->category->id,
            'title' => 'Pay Electricity Bill',
            'description' => 'Monthly electricity bill payment',
            'frequency_type' => 'one_time',
            'frequency_config' => ['time_slots' => ['10:00']],
            'start_date' => now()->addDays(1)->format('Y-m-d'),
            'priority' => 'high',
            'amount' => 150.50,
        ]);

        $response->assertStatus(201)
            ->assertJsonStructure([
                'message',
                'notification' => ['id', 'title', 'frequency_type', 'priority', 'amount', 'category'],
            ]);

        $this->assertDatabaseHas('notifications', ['title' => 'Pay Electricity Bill']);
    }

    public function test_can_create_daily_notification(): void
    {
        $response = $this->actingAs($this->user)->postJson('/api/notifications', [
            'category_id' => $this->category->id,
            'title' => 'Take Medicine',
            'frequency_type' => 'daily',
            'frequency_config' => ['time_slots' => ['08:00', '14:00', '20:00']],
            'start_date' => now()->format('Y-m-d'),
            'priority' => 'critical',
        ]);

        $response->assertStatus(201);
    }

    public function test_can_create_weekly_notification(): void
    {
        $response = $this->actingAs($this->user)->postJson('/api/notifications', [
            'category_id' => $this->category->id,
            'title' => 'Team Meeting',
            'frequency_type' => 'weekly',
            'frequency_config' => ['selected_days' => [1, 3, 5], 'time_slots' => ['10:00']],
            'start_date' => now()->format('Y-m-d'),
            'priority' => 'medium',
        ]);

        $response->assertStatus(201);
    }

    public function test_can_create_monthly_notification(): void
    {
        $response = $this->actingAs($this->user)->postJson('/api/notifications', [
            'category_id' => $this->category->id,
            'title' => 'Credit Card Payment',
            'frequency_type' => 'monthly',
            'frequency_config' => ['day_of_month' => 15, 'time_slots' => ['09:00']],
            'start_date' => now()->format('Y-m-d'),
            'priority' => 'high',
            'amount' => 5000,
        ]);

        $response->assertStatus(201);
    }

    public function test_can_list_notifications(): void
    {
        Notification::factory()->count(3)->create([
            'user_id' => $this->user->id,
            'category_id' => $this->category->id,
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/notifications');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'notifications',
                'meta' => ['current_page', 'last_page', 'per_page', 'total'],
            ]);
    }

    public function test_can_filter_notifications_by_category(): void
    {
        Notification::factory()->count(2)->create([
            'user_id' => $this->user->id,
            'category_id' => $this->category->id,
        ]);

        $response = $this->actingAs($this->user)
            ->getJson("/api/notifications?category_id={$this->category->id}");

        $response->assertStatus(200);
        $this->assertEquals(2, $response->json('meta.total'));
    }

    public function test_can_filter_notifications_by_priority(): void
    {
        Notification::factory()->create([
            'user_id' => $this->user->id,
            'category_id' => $this->category->id,
            'priority' => 'critical',
        ]);

        $response = $this->actingAs($this->user)
            ->getJson('/api/notifications?priority=critical');

        $response->assertStatus(200);
        $this->assertEquals(1, $response->json('meta.total'));
    }

    public function test_can_search_notifications(): void
    {
        Notification::factory()->create([
            'user_id' => $this->user->id,
            'category_id' => $this->category->id,
            'title' => 'Electricity Bill Payment',
        ]);

        $response = $this->actingAs($this->user)
            ->getJson('/api/notifications?search=Electricity');

        $response->assertStatus(200);
        $this->assertEquals(1, $response->json('meta.total'));
    }

    public function test_can_show_notification(): void
    {
        $notification = Notification::factory()->create([
            'user_id' => $this->user->id,
            'category_id' => $this->category->id,
        ]);

        $response = $this->actingAs($this->user)
            ->getJson("/api/notifications/{$notification->id}");

        $response->assertStatus(200)
            ->assertJsonStructure(['notification' => ['id', 'title', 'category']]);
    }

    public function test_cannot_view_other_users_notification(): void
    {
        $otherUser = User::factory()->create();
        $notification = Notification::factory()->create([
            'user_id' => $otherUser->id,
            'category_id' => $this->category->id,
        ]);

        $response = $this->actingAs($this->user)
            ->getJson("/api/notifications/{$notification->id}");

        $response->assertStatus(403);
    }

    public function test_can_update_notification(): void
    {
        $notification = Notification::factory()->create([
            'user_id' => $this->user->id,
            'category_id' => $this->category->id,
            'title' => 'Old Title',
        ]);

        $response = $this->actingAs($this->user)
            ->putJson("/api/notifications/{$notification->id}", [
                'title' => 'New Title',
                'priority' => 'critical',
            ]);

        $response->assertStatus(200);
        $this->assertEquals('New Title', $notification->fresh()->title);
    }

    public function test_can_delete_notification(): void
    {
        $notification = Notification::factory()->create([
            'user_id' => $this->user->id,
            'category_id' => $this->category->id,
        ]);

        $response = $this->actingAs($this->user)
            ->deleteJson("/api/notifications/{$notification->id}");

        $response->assertStatus(200);
        $this->assertSoftDeleted('notifications', ['id' => $notification->id]);
    }

    public function test_can_toggle_notification(): void
    {
        $notification = Notification::factory()->create([
            'user_id' => $this->user->id,
            'category_id' => $this->category->id,
            'is_active' => true,
        ]);

        $response = $this->actingAs($this->user)
            ->patchJson("/api/notifications/{$notification->id}/toggle");

        $response->assertStatus(200);
        $this->assertFalse($notification->fresh()->is_active);
    }

    public function test_validates_notification_fields(): void
    {
        $response = $this->actingAs($this->user)->postJson('/api/notifications', []);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['category_id', 'title', 'frequency_type', 'start_date']);
    }
}
