<?php

namespace Tests\Feature\Api;

use App\Models\Category;
use App\Models\Notification;
use App\Models\NotificationOccurrence;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class DashboardTest extends TestCase
{
    use RefreshDatabase;

    private User $user;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create(['email_verified_at' => now()]);
        $this->seed(\Database\Seeders\CategorySeeder::class);
    }

    private function createNotificationWithOccurrence(string $status = 'pending', ?string $scheduledAt = null): void
    {
        $category = Category::where('is_default', true)->first();
        $notification = Notification::factory()->create([
            'user_id' => $this->user->id,
            'category_id' => $category->id,
            'is_active' => true,
        ]);

        NotificationOccurrence::create([
            'notification_id' => $notification->id,
            'scheduled_at' => $scheduledAt ?? now()->addHour(),
            'status' => $status,
            'acted_at' => $status === 'completed' ? now() : null,
        ]);
    }

    public function test_today_endpoint(): void
    {
        // Create occurrence for today
        $this->createNotificationWithOccurrence('pending', now()->addHours(2)->toDateTimeString());

        $response = $this->actingAs($this->user)->getJson('/api/dashboard/today');

        $response->assertStatus(200)
            ->assertJsonStructure(['date', 'occurrences', 'total']);
    }

    public function test_summary_endpoint(): void
    {
        $this->createNotificationWithOccurrence('pending', now()->addHour()->toDateTimeString());
        $this->createNotificationWithOccurrence('completed', now()->subHour()->toDateTimeString());
        $this->createNotificationWithOccurrence('pending', now()->subHour()->toDateTimeString());

        $response = $this->actingAs($this->user)->getJson('/api/dashboard/summary');

        $response->assertStatus(200)
            ->assertJsonStructure(['total_active', 'due_today', 'overdue', 'completed_today']);
    }

    public function test_calendar_endpoint(): void
    {
        $this->createNotificationWithOccurrence('pending', now()->addDays(5)->toDateTimeString());

        $response = $this->actingAs($this->user)->getJson(
            '/api/dashboard/calendar?month=' . now()->month . '&year=' . now()->year
        );

        $response->assertStatus(200)
            ->assertJsonStructure(['month', 'year', 'days']);
    }

    public function test_upcoming_endpoint(): void
    {
        $this->createNotificationWithOccurrence('pending', now()->addDays(2)->toDateTimeString());
        $this->createNotificationWithOccurrence('pending', now()->addDays(5)->toDateTimeString());

        $response = $this->actingAs($this->user)->getJson('/api/dashboard/upcoming?days=7');

        $response->assertStatus(200)
            ->assertJsonStructure(['days', 'upcoming']);
    }

    public function test_summary_returns_zeros_for_new_user(): void
    {
        $response = $this->actingAs($this->user)->getJson('/api/dashboard/summary');

        $response->assertStatus(200)
            ->assertJson([
                'total_active' => 0,
                'due_today' => 0,
                'overdue' => 0,
                'completed_today' => 0,
            ]);
    }
}
