<?php

namespace Tests\Feature\Api;

use App\Models\Category;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class CategoryTest extends TestCase
{
    use RefreshDatabase;

    private User $user;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create(['email_verified_at' => now()]);

        // Seed default categories
        $this->seed(\Database\Seeders\CategorySeeder::class);
    }

    public function test_can_list_categories(): void
    {
        $response = $this->actingAs($this->user)->getJson('/api/categories');

        $response->assertStatus(200)
            ->assertJsonStructure([
                'categories' => [
                    '*' => ['id', 'name', 'icon', 'color', 'is_default', 'sort_order'],
                ],
            ]);

        // Should include default categories
        $this->assertGreaterThanOrEqual(7, count($response->json('categories')));
    }

    public function test_can_create_custom_category(): void
    {
        $response = $this->actingAs($this->user)->postJson('/api/categories', [
            'name' => 'My Custom',
            'icon' => 'star',
            'color' => '#FF5722',
        ]);

        $response->assertStatus(201)
            ->assertJson([
                'category' => [
                    'name' => 'My Custom',
                    'icon' => 'star',
                    'color' => '#FF5722',
                    'is_default' => false,
                ],
            ]);
    }

    public function test_can_update_custom_category(): void
    {
        $category = Category::create([
            'user_id' => $this->user->id,
            'name' => 'Old Name',
            'icon' => 'star',
            'color' => '#FF5722',
        ]);

        $response = $this->actingAs($this->user)->putJson("/api/categories/{$category->id}", [
            'name' => 'New Name',
        ]);

        $response->assertStatus(200);
        $this->assertEquals('New Name', $category->fresh()->name);
    }

    public function test_cannot_update_default_category(): void
    {
        $defaultCategory = Category::where('is_default', true)->first();

        $response = $this->actingAs($this->user)->putJson("/api/categories/{$defaultCategory->id}", [
            'name' => 'Hacked Name',
        ]);

        $response->assertStatus(403);
    }

    public function test_can_delete_custom_category(): void
    {
        $category = Category::create([
            'user_id' => $this->user->id,
            'name' => 'To Delete',
        ]);

        $response = $this->actingAs($this->user)->deleteJson("/api/categories/{$category->id}");

        $response->assertStatus(200);
        $this->assertDatabaseMissing('categories', ['id' => $category->id]);
    }

    public function test_cannot_delete_default_category(): void
    {
        $defaultCategory = Category::where('is_default', true)->first();

        $response = $this->actingAs($this->user)->deleteJson("/api/categories/{$defaultCategory->id}");

        $response->assertStatus(403);
    }

    public function test_cannot_modify_other_users_category(): void
    {
        $otherUser = User::factory()->create();
        $category = Category::create([
            'user_id' => $otherUser->id,
            'name' => 'Other User Category',
        ]);

        $response = $this->actingAs($this->user)->putJson("/api/categories/{$category->id}", [
            'name' => 'Stolen',
        ]);

        $response->assertStatus(403);
    }

    public function test_can_reorder_categories(): void
    {
        $categories = Category::where('is_default', true)->get();

        $reorderData = $categories->take(3)->map(function ($cat, $index) {
            return ['id' => $cat->id, 'sort_order' => $index + 10];
        })->toArray();

        $response = $this->actingAs($this->user)->putJson('/api/categories/reorder', [
            'categories' => $reorderData,
        ]);

        $response->assertStatus(200);
    }
}
