<?php

namespace App\Services;

use App\Models\Notification;
use App\Models\NotificationOccurrence;
use Carbon\Carbon;
use Illuminate\Support\Collection;

class OccurrenceService
{
    public function generateOccurrences(Notification $notification, int $daysAhead = 30): void
    {
        $userTimezone = $notification->user->getTimezone();

        // Delete future pending occurrences before regenerating
        $notification->occurrences()
            ->where('status', 'pending')
            ->where('scheduled_at', '>', now())
            ->delete();

        $dates = $this->calculateDates($notification, $daysAhead, $userTimezone);

        foreach ($dates as $date) {
            $timeSlots = $this->getTimeSlots($notification);
            foreach ($timeSlots as $time) {
                // Create datetime in user's timezone, then convert to UTC for storage
                $scheduledAt = Carbon::createFromFormat(
                    'Y-m-d H:i',
                    $date->format('Y-m-d') . ' ' . $time,
                    $userTimezone
                )->utc();

                if ($scheduledAt->isPast()) {
                    continue;
                }

                // Avoid duplicates
                $exists = $notification->occurrences()
                    ->where('scheduled_at', $scheduledAt)
                    ->exists();

                if (!$exists) {
                    NotificationOccurrence::create([
                        'notification_id' => $notification->id,
                        'scheduled_at' => $scheduledAt,
                        'status' => 'pending',
                    ]);
                }
            }
        }
    }

    private function calculateDates(Notification $notification, int $daysAhead, string $userTimezone): Collection
    {
        $startDate = Carbon::parse($notification->start_date);
        $endDate = $notification->end_date
            ? Carbon::parse($notification->end_date)
            : now()->addDays($daysAhead);

        // Don't go beyond daysAhead from today
        $maxDate = now()->addDays($daysAhead);
        if ($endDate->gt($maxDate)) {
            $endDate = $maxDate;
        }

        // "Today" in the user's timezone
        $userToday = now()->setTimezone($userTimezone)->startOfDay();

        $dates = collect();
        $config = $notification->frequency_config ?? [];

        switch ($notification->frequency_type) {
            case 'one_time':
                if ($startDate->gte($userToday)) {
                    $dates->push($startDate);
                }
                break;

            case 'daily':
                $current = $startDate->copy()->max($userToday);
                while ($current->lte($endDate)) {
                    $dates->push($current->copy());
                    $current->addDay();
                }
                break;

            case 'weekly':
                $selectedDays = $config['selected_days'] ?? [1]; // Default Monday
                $current = $startDate->copy()->max($userToday);
                while ($current->lte($endDate)) {
                    if (in_array($current->dayOfWeekIso, $selectedDays)) {
                        $dates->push($current->copy());
                    }
                    $current->addDay();
                }
                break;

            case 'bi_weekly':
                $selectedDays = $config['selected_days'] ?? [1];
                $current = $startDate->copy()->max($userToday);
                $weekCounter = 0;
                $lastWeek = null;
                while ($current->lte($endDate)) {
                    $currentWeek = $current->weekOfYear;
                    if ($lastWeek !== null && $currentWeek !== $lastWeek) {
                        $weekCounter++;
                    }
                    $lastWeek = $currentWeek;

                    if ($weekCounter % 2 === 0 && in_array($current->dayOfWeekIso, $selectedDays)) {
                        $dates->push($current->copy());
                    }
                    $current->addDay();
                }
                break;

            case 'monthly':
                $dayOfMonth = $config['day_of_month'] ?? $startDate->day;
                $current = $startDate->copy()->max($userToday)->startOfMonth();
                while ($current->lte($endDate)) {
                    $targetDay = min($dayOfMonth, $current->daysInMonth);
                    $date = $current->copy()->day($targetDay);
                    if ($date->gte($userToday) && $date->lte($endDate)) {
                        $dates->push($date);
                    }
                    $current->addMonth();
                }
                break;

            case 'quarterly':
                $dayOfMonth = $config['day_of_month'] ?? $startDate->day;
                $current = $startDate->copy()->max($userToday)->startOfMonth();
                while ($current->lte($endDate)) {
                    $targetDay = min($dayOfMonth, $current->daysInMonth);
                    $date = $current->copy()->day($targetDay);
                    if ($date->gte($userToday) && $date->lte($endDate)) {
                        $dates->push($date);
                    }
                    $current->addMonths(3);
                }
                break;

            case 'yearly':
                $current = $startDate->copy()->max($userToday);
                while ($current->lte($endDate)) {
                    if ($current->gte($userToday)) {
                        $dates->push($current->copy());
                    }
                    $current->addYear();
                }
                break;

            case 'custom':
                $interval = $config['interval_days'] ?? 1;
                $current = $startDate->copy()->max($userToday);
                while ($current->lte($endDate)) {
                    $dates->push($current->copy());
                    $current->addDays($interval);
                }
                break;

            case 'weekdays':
                $current = $startDate->copy()->max($userToday);
                while ($current->lte($endDate)) {
                    if ($current->isWeekday()) {
                        $dates->push($current->copy());
                    }
                    $current->addDay();
                }
                break;

            case 'weekends':
                $current = $startDate->copy()->max($userToday);
                while ($current->lte($endDate)) {
                    if ($current->isWeekend()) {
                        $dates->push($current->copy());
                    }
                    $current->addDay();
                }
                break;
        }

        return $dates;
    }

    private function getTimeSlots(Notification $notification): array
    {
        $config = $notification->frequency_config ?? [];
        return $config['time_slots'] ?? ['09:00'];
    }
}
