<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\SnoozeOccurrenceRequest;
use App\Http\Resources\OccurrenceResource;
use App\Models\NotificationOccurrence;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class OccurrenceController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $query = NotificationOccurrence::whereHas('notification', function ($q) use ($request) {
            $q->where('user_id', $request->user()->id);
        })->with(['notification.category']);

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('notification_id')) {
            $query->where('notification_id', $request->notification_id);
        }

        if ($request->has('from_date')) {
            $from = Carbon::parse($request->from_date, $request->user()->getTimezone())->startOfDay()->utc();
            $query->where('scheduled_at', '>=', $from);
        }

        if ($request->has('to_date')) {
            $to = Carbon::parse($request->to_date, $request->user()->getTimezone())->endOfDay()->utc();
            $query->where('scheduled_at', '<=', $to);
        }

        $occurrences = $query->orderBy('scheduled_at')
            ->paginate($request->input('per_page', 20));

        return response()->json([
            'occurrences' => OccurrenceResource::collection($occurrences),
            'meta' => [
                'current_page' => $occurrences->currentPage(),
                'last_page' => $occurrences->lastPage(),
                'per_page' => $occurrences->perPage(),
                'total' => $occurrences->total(),
            ],
        ]);
    }

    public function complete(Request $request, NotificationOccurrence $occurrence): JsonResponse
    {
        if ($occurrence->notification->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized.'], 403);
        }

        $occurrence->update([
            'status' => 'completed',
            'acted_at' => now(),
        ]);

        return response()->json([
            'message' => 'Marked as completed.',
            'occurrence' => new OccurrenceResource($occurrence->fresh()),
        ]);
    }

    public function dismiss(Request $request, NotificationOccurrence $occurrence): JsonResponse
    {
        if ($occurrence->notification->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized.'], 403);
        }

        $occurrence->update([
            'status' => 'dismissed',
            'acted_at' => now(),
        ]);

        return response()->json([
            'message' => 'Occurrence dismissed.',
            'occurrence' => new OccurrenceResource($occurrence->fresh()),
        ]);
    }

    public function snooze(SnoozeOccurrenceRequest $request, NotificationOccurrence $occurrence): JsonResponse
    {
        if ($occurrence->notification->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized.'], 403);
        }

        $snoozedUntil = now()->addMinutes($request->duration_minutes);

        $occurrence->update([
            'status' => 'snoozed',
            'snoozed_until' => $snoozedUntil,
        ]);

        return response()->json([
            'message' => 'Snoozed until ' . $snoozedUntil->toISOString(),
            'occurrence' => new OccurrenceResource($occurrence->fresh()),
        ]);
    }

    public function skip(Request $request, NotificationOccurrence $occurrence): JsonResponse
    {
        if ($occurrence->notification->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized.'], 403);
        }

        $occurrence->update([
            'status' => 'skipped',
            'acted_at' => now(),
        ]);

        return response()->json([
            'message' => 'Occurrence skipped.',
            'occurrence' => new OccurrenceResource($occurrence->fresh()),
        ]);
    }
}
