<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreNotificationRequest;
use App\Http\Requests\UpdateNotificationRequest;
use App\Http\Resources\NotificationResource;
use App\Models\Notification;
use App\Services\OccurrenceService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class NotificationController extends Controller
{
    public function __construct(private OccurrenceService $occurrenceService)
    {
    }

    public function index(Request $request): JsonResponse
    {
        $query = Notification::where('user_id', $request->user()->id)
            ->with(['category', 'attachments']);

        // Filters
        if ($request->has('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        if ($request->has('priority')) {
            $query->where('priority', $request->priority);
        }

        if ($request->has('is_active')) {
            $query->where('is_active', filter_var($request->is_active, FILTER_VALIDATE_BOOLEAN));
        }

        if ($request->has('frequency_type')) {
            $query->where('frequency_type', $request->frequency_type);
        }

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        $sortBy = $request->input('sort_by', 'created_at');
        $sortDir = $request->input('sort_dir', 'desc');
        $query->orderBy($sortBy, $sortDir);

        $notifications = $query->paginate($request->input('per_page', 15));

        return response()->json([
            'notifications' => NotificationResource::collection($notifications),
            'meta' => [
                'current_page' => $notifications->currentPage(),
                'last_page' => $notifications->lastPage(),
                'per_page' => $notifications->perPage(),
                'total' => $notifications->total(),
            ],
        ]);
    }

    public function store(StoreNotificationRequest $request): JsonResponse
    {
        $notification = Notification::create([
            'user_id' => $request->user()->id,
            ...$request->validated(),
        ]);

        $this->occurrenceService->generateOccurrences($notification);

        $notification->load(['category', 'attachments']);

        return response()->json([
            'message' => 'Notification created successfully.',
            'notification' => new NotificationResource($notification),
        ], 201);
    }

    public function show(Request $request, Notification $notification): JsonResponse
    {
        if ($notification->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized.'], 403);
        }

        $notification->load(['category', 'attachments', 'occurrences' => function ($q) {
            $q->orderBy('scheduled_at')->limit(10);
        }]);

        return response()->json([
            'notification' => new NotificationResource($notification),
        ]);
    }

    public function update(UpdateNotificationRequest $request, Notification $notification): JsonResponse
    {
        if ($notification->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized.'], 403);
        }

        $notification->update($request->validated());

        // Regenerate occurrences if scheduling changed
        $scheduleFields = ['frequency_type', 'frequency_config', 'start_date', 'end_date'];
        if ($request->hasAny($scheduleFields)) {
            $this->occurrenceService->generateOccurrences($notification);
        }

        $notification->load(['category', 'attachments']);

        return response()->json([
            'message' => 'Notification updated successfully.',
            'notification' => new NotificationResource($notification->fresh()),
        ]);
    }

    public function destroy(Request $request, Notification $notification): JsonResponse
    {
        if ($notification->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized.'], 403);
        }

        $notification->delete(); // Soft delete

        return response()->json([
            'message' => 'Notification deleted successfully.',
        ]);
    }

    public function toggle(Request $request, Notification $notification): JsonResponse
    {
        if ($notification->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized.'], 403);
        }

        $notification->update(['is_active' => !$notification->is_active]);

        return response()->json([
            'message' => $notification->is_active ? 'Notification activated.' : 'Notification deactivated.',
            'is_active' => $notification->is_active,
        ]);
    }

    public function skipNext(Request $request, Notification $notification): JsonResponse
    {
        if ($notification->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Unauthorized.'], 403);
        }

        $nextOccurrence = $notification->occurrences()
            ->where('status', 'pending')
            ->where('scheduled_at', '>', now())
            ->orderBy('scheduled_at')
            ->first();

        if (!$nextOccurrence) {
            return response()->json([
                'message' => 'No upcoming occurrence to skip.',
            ], 404);
        }

        $nextOccurrence->update(['status' => 'skipped']);

        return response()->json([
            'message' => 'Next occurrence skipped.',
            'skipped_at' => $nextOccurrence->scheduled_at->toISOString(),
        ]);
    }
}
