<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\OccurrenceResource;
use App\Models\Notification;
use App\Models\NotificationOccurrence;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    /**
     * Get start/end of "today" in UTC based on the user's timezone.
     */
    private function getUserDayBoundaries($user): array
    {
        $tz = $user->getTimezone();
        return [
            Carbon::now($tz)->startOfDay()->utc(),
            Carbon::now($tz)->endOfDay()->utc(),
        ];
    }

    public function today(Request $request): JsonResponse
    {
        $tz = $request->user()->getTimezone();
        [$today, $endOfDay] = $this->getUserDayBoundaries($request->user());

        $occurrences = NotificationOccurrence::whereHas('notification', function ($q) use ($request) {
            $q->where('user_id', $request->user()->id)->where('is_active', true);
        })
            ->with(['notification.category'])
            ->whereBetween('scheduled_at', [$today, $endOfDay])
            ->orderBy('scheduled_at')
            ->get();

        return response()->json([
            'date' => Carbon::now($tz)->format('Y-m-d'),
            'occurrences' => OccurrenceResource::collection($occurrences),
            'total' => $occurrences->count(),
        ]);
    }

    public function summary(Request $request): JsonResponse
    {
        $userId = $request->user()->id;
        [$startOfDay, $endOfDay] = $this->getUserDayBoundaries($request->user());

        $totalActive = Notification::where('user_id', $userId)
            ->where('is_active', true)
            ->count();

        $dueToday = NotificationOccurrence::whereHas('notification', function ($q) use ($userId) {
            $q->where('user_id', $userId)->where('is_active', true);
        })
            ->whereBetween('scheduled_at', [$startOfDay, $endOfDay])
            ->where('status', 'pending')
            ->count();

        // Overdue uses raw now() — past the actual moment, timezone-independent
        $overdue = NotificationOccurrence::whereHas('notification', function ($q) use ($userId) {
            $q->where('user_id', $userId)->where('is_active', true);
        })
            ->where('scheduled_at', '<', now())
            ->where('status', 'pending')
            ->count();

        $completedToday = NotificationOccurrence::whereHas('notification', function ($q) use ($userId) {
            $q->where('user_id', $userId);
        })
            ->where('status', 'completed')
            ->whereBetween('acted_at', [$startOfDay, $endOfDay])
            ->count();

        return response()->json([
            'total_active' => $totalActive,
            'due_today' => $dueToday,
            'overdue' => $overdue,
            'completed_today' => $completedToday,
        ]);
    }

    public function calendar(Request $request): JsonResponse
    {
        $tz = $request->user()->getTimezone();
        $month = $request->input('month', Carbon::now($tz)->month);
        $year = $request->input('year', Carbon::now($tz)->year);

        // Compute month boundaries in user's timezone, convert to UTC for query
        $startOfMonth = Carbon::create($year, $month, 1, 0, 0, 0, $tz)->startOfMonth()->utc();
        $endOfMonth = Carbon::create($year, $month, 1, 0, 0, 0, $tz)->endOfMonth()->endOfDay()->utc();

        $occurrences = NotificationOccurrence::whereHas('notification', function ($q) use ($request) {
            $q->where('user_id', $request->user()->id)->where('is_active', true);
        })
            ->with(['notification.category'])
            ->whereBetween('scheduled_at', [$startOfMonth, $endOfMonth])
            ->orderBy('scheduled_at')
            ->get();

        // Group by date in user's timezone
        $grouped = $occurrences->groupBy(function ($occ) use ($tz) {
            return $occ->scheduled_at->setTimezone($tz)->format('Y-m-d');
        })->map(function ($dayOccurrences) {
            return [
                'count' => $dayOccurrences->count(),
                'categories' => $dayOccurrences->map(function ($occ) {
                    return $occ->notification?->category?->color;
                })->filter()->unique()->values(),
                'occurrences' => OccurrenceResource::collection($dayOccurrences),
            ];
        });

        return response()->json([
            'month' => $month,
            'year' => $year,
            'days' => $grouped,
        ]);
    }

    public function upcoming(Request $request): JsonResponse
    {
        $tz = $request->user()->getTimezone();
        $days = (int) $request->input('days', 7);

        $endBoundary = Carbon::now($tz)->addDays($days)->endOfDay()->utc();

        $occurrences = NotificationOccurrence::whereHas('notification', function ($q) use ($request) {
            $q->where('user_id', $request->user()->id)->where('is_active', true);
        })
            ->with(['notification.category'])
            ->where('scheduled_at', '>=', now())
            ->where('scheduled_at', '<=', $endBoundary)
            ->where('status', 'pending')
            ->orderBy('scheduled_at')
            ->get();

        // Group by date in user's timezone
        $grouped = $occurrences->groupBy(function ($occ) use ($tz) {
            return $occ->scheduled_at->setTimezone($tz)->format('Y-m-d');
        })->map(function ($dayOccurrences, $date) {
            return [
                'date' => $date,
                'occurrences' => OccurrenceResource::collection($dayOccurrences),
            ];
        })->values();

        return response()->json([
            'days' => $days,
            'upcoming' => $grouped,
        ]);
    }
}
