<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\ReorderCategoriesRequest;
use App\Http\Requests\StoreCategoryRequest;
use App\Http\Requests\UpdateCategoryRequest;
use App\Http\Resources\CategoryResource;
use App\Models\Category;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class CategoryController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $categories = Category::forUser($request->user()->id)
            ->withCount('notifications')
            ->orderBy('sort_order')
            ->get();

        return response()->json([
            'categories' => CategoryResource::collection($categories),
        ]);
    }

    public function store(StoreCategoryRequest $request): JsonResponse
    {
        $maxOrder = Category::forUser($request->user()->id)->max('sort_order') ?? 0;

        $category = Category::create([
            'user_id' => $request->user()->id,
            'name' => $request->name,
            'icon' => $request->input('icon', 'notifications'),
            'color' => $request->input('color', '#2196F3'),
            'sort_order' => $maxOrder + 1,
        ]);

        return response()->json([
            'message' => 'Category created successfully.',
            'category' => new CategoryResource($category),
        ], 201);
    }

    public function update(UpdateCategoryRequest $request, Category $category): JsonResponse
    {
        if ($category->is_default) {
            return response()->json([
                'message' => 'Cannot modify default categories.',
            ], 403);
        }

        if ($category->user_id !== $request->user()->id) {
            return response()->json([
                'message' => 'Unauthorized.',
            ], 403);
        }

        $category->update($request->validated());

        return response()->json([
            'message' => 'Category updated successfully.',
            'category' => new CategoryResource($category->fresh()),
        ]);
    }

    public function destroy(Request $request, Category $category): JsonResponse
    {
        if ($category->is_default) {
            return response()->json([
                'message' => 'Cannot delete default categories.',
            ], 403);
        }

        if ($category->user_id !== $request->user()->id) {
            return response()->json([
                'message' => 'Unauthorized.',
            ], 403);
        }

        $category->delete();

        return response()->json([
            'message' => 'Category deleted successfully.',
        ]);
    }

    public function reorder(ReorderCategoriesRequest $request): JsonResponse
    {
        foreach ($request->categories as $item) {
            Category::where('id', $item['id'])
                ->where(function ($q) use ($request) {
                    $q->whereNull('user_id')->orWhere('user_id', $request->user()->id);
                })
                ->update(['sort_order' => $item['sort_order']]);
        }

        return response()->json([
            'message' => 'Categories reordered successfully.',
        ]);
    }
}
