<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\ChangePasswordRequest;
use App\Http\Requests\LoginRequest;
use App\Http\Requests\RegisterRequest;
use App\Http\Requests\ResetPasswordRequest;
use App\Http\Requests\SendOtpRequest;
use App\Http\Requests\UpdateProfileRequest;
use App\Http\Requests\VerifyOtpRequest;
use App\Http\Resources\UserResource;
use App\Models\User;
use App\Services\OccurrenceService;
use App\Services\OtpService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class AuthController extends Controller
{
    public function __construct(private OtpService $otpService)
    {
    }

    public function register(RegisterRequest $request): JsonResponse
    {
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => $request->password,
        ]);

        $this->otpService->generate($user->email, 'signup', $user->id);

        return response()->json([
            'message' => 'Registration successful. Please verify your email with the OTP sent.',
            'user' => new UserResource($user),
        ], 201);
    }

    public function verifyOtp(VerifyOtpRequest $request): JsonResponse
    {
        $otp = $this->otpService->verify($request->email, $request->code, $request->type);

        if (!$otp) {
            return response()->json([
                'message' => 'Invalid or expired OTP.',
            ], 422);
        }

        if ($request->type === 'signup') {
            $user = User::where('email', $request->email)->first();
            if ($user) {
                $user->forceFill(['email_verified_at' => now()])->save();
                $token = $user->createToken('auth-token')->plainTextToken;

                return response()->json([
                    'message' => 'Email verified successfully.',
                    'user' => new UserResource($user->fresh()),
                    'token' => $token,
                ]);
            }
        }

        return response()->json([
            'message' => 'OTP verified successfully.',
        ]);
    }

    public function login(LoginRequest $request): JsonResponse
    {
        $user = User::where('email', $request->email)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json([
                'message' => 'Invalid credentials.',
            ], 401);
        }

        if (!$user->email_verified_at) {
            return response()->json([
                'message' => 'Please verify your email first.',
                'email_verified' => false,
            ], 403);
        }

        $token = $user->createToken('auth-token')->plainTextToken;

        return response()->json([
            'message' => 'Login successful.',
            'user' => new UserResource($user),
            'token' => $token,
        ]);
    }

    public function sendOtp(SendOtpRequest $request): JsonResponse
    {
        $user = User::where('email', $request->email)->first();

        if ($request->type === 'signup' && !$user) {
            return response()->json([
                'message' => 'No account found with this email.',
            ], 404);
        }

        if ($request->type === 'password_reset' && !$user) {
            return response()->json([
                'message' => 'No account found with this email.',
            ], 404);
        }

        $this->otpService->generate($request->email, $request->type, $user?->id);

        return response()->json([
            'message' => 'OTP sent to your email.',
        ]);
    }

    public function forgotPassword(Request $request): JsonResponse
    {
        $request->validate(['email' => 'required|email']);

        $user = User::where('email', $request->email)->first();

        if (!$user) {
            return response()->json([
                'message' => 'No account found with this email.',
            ], 404);
        }

        $this->otpService->generate($request->email, 'password_reset', $user->id);

        return response()->json([
            'message' => 'Password reset OTP sent to your email.',
        ]);
    }

    public function resetPassword(ResetPasswordRequest $request): JsonResponse
    {
        $otp = $this->otpService->verify($request->email, $request->code, 'password_reset');

        if (!$otp) {
            return response()->json([
                'message' => 'Invalid or expired OTP.',
            ], 422);
        }

        $user = User::where('email', $request->email)->first();

        if (!$user) {
            return response()->json([
                'message' => 'No account found with this email.',
            ], 404);
        }

        $user->update(['password' => $request->password]);

        // Revoke all tokens
        $user->tokens()->delete();

        return response()->json([
            'message' => 'Password reset successfully. Please login with your new password.',
        ]);
    }

    public function logout(Request $request): JsonResponse
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'message' => 'Logged out successfully.',
        ]);
    }

    public function profile(Request $request): JsonResponse
    {
        return response()->json([
            'user' => new UserResource($request->user()),
        ]);
    }

    public function updateProfile(UpdateProfileRequest $request): JsonResponse
    {
        $user = $request->user();
        $data = $request->validated();

        // Merge preferences instead of replacing to preserve existing keys
        if (isset($data['preferences'])) {
            $oldTimezone = $user->getTimezone();
            $data['preferences'] = array_merge($user->preferences ?? [], $data['preferences']);
            $user->update($data);

            // Regenerate occurrences if timezone changed
            $newTimezone = $user->fresh()->getTimezone();
            if ($oldTimezone !== $newTimezone) {
                $occurrenceService = app(OccurrenceService::class);
                $user->notifications()->where('is_active', true)->each(function ($notification) use ($occurrenceService) {
                    $occurrenceService->generateOccurrences($notification);
                });
            }
        } else {
            $user->update($data);
        }

        return response()->json([
            'message' => 'Profile updated successfully.',
            'user' => new UserResource($user->fresh()),
        ]);
    }

    public function changePassword(ChangePasswordRequest $request): JsonResponse
    {
        $request->user()->update([
            'password' => $request->password,
        ]);

        return response()->json([
            'message' => 'Password changed successfully.',
        ]);
    }

    public function deleteAccount(Request $request): JsonResponse
    {
        $user = $request->user();
        $user->tokens()->delete();
        $user->delete();

        return response()->json([
            'message' => 'Account deleted successfully.',
        ]);
    }
}
